# -*- coding: utf-8 -*-
"""

@author: dyrvik
"""
import numpy as np
import matplotlib.pyplot as plt
import matplotlib.cm as cm
import matplotlib as mpl
import seaborn as sns
import pandas as pd
import os
from mpl_toolkits.axes_grid1 import Divider, Size

"""Colourmaps"""
CM_SUBSECTION_GREENS = np.linspace(0.4, 1.0, 1000) 
COLOURS_GREENS = [ cm.Greens(x) for x in CM_SUBSECTION_GREENS ]

CM_SUBSECTION_ORANGES = np.linspace(0.3, 1.0, 1000) 
COLOURS_ORANGES = [ cm.Oranges(x) for x in CM_SUBSECTION_ORANGES ]

"""File handling"""
main_folder_path = os.path.dirname(os.path.abspath(__file__))
main_folder_path = '\\\\?\\' + main_folder_path
data_folder_path = main_folder_path
export_folder_path = main_folder_path
# =============================================================================
# export_folder_name = 'Plots'
# export_folder_path = os.path.join(main_folder_path,export_folder_name)
# if not os.path.exists(export_folder_path):
#     os.mkdir(export_folder_path) # create folder if non-existent
# =============================================================================

file_name = 'thicknesses.csv'
file_path = os.path.join(data_folder_path,file_name)
df = pd.read_csv(file_path) 

df_prof_glass = df[(df['Substrate'] == 'Glass') & (df['Method'] == 'Profilometry')]
df_xrr_si = df[(df['Substrate'] == 'Si') & (df['Method'] == 'XRR')]
df_ellip_si = df[(df['Substrate'] == 'Si') & (df['Method'] == 'Ellipsometry')]
df_ellip_ito = df[(df['Substrate'] == 'ITO') & (df['Method'] == 'Ellipsometry')]

"""Some constants for graphical settings"""
lw = 3
ticksize_major = 8
ticksize_minor = 8


"""Plotting the box swarm plot"""
fig = plt.figure(figsize=(7.5, 7.5))
h = [Size.Fixed(1.0), Size.Scaled(1.), Size.Fixed(.2)]
v = [Size.Fixed(0.7), Size.Scaled(1.), Size.Fixed(.5)]
divider = Divider(fig, (0, 0, 1, 1), h, v, aspect=False)
# The width and height of the rectangle are ignored.
ax = plt.axes([0.15, 0.15, 0.85, 0.85])

ax.scatter(df_prof_glass['TFB concentration [mg/ml]'], df_prof_glass['Thickness [nm]'], s=200, color='w', marker='s', edgecolor=COLOURS_ORANGES[400], linewidth=lw, label='on glass, profilometry')
ax.scatter(df_xrr_si['TFB concentration [mg/ml]'], df_xrr_si['Thickness [nm]'], s=200, color='w', marker='^', edgecolor=COLOURS_GREENS[200], linewidth=lw, label='on Si, XRR')
ax.scatter(df_ellip_si['TFB concentration [mg/ml]'], df_ellip_si['Thickness [nm]'], s=200, color='w', marker='o', edgecolor=COLOURS_ORANGES[800], linewidth=lw, label='on Si, ellipsometry')
ax.scatter(df_ellip_ito['TFB concentration [mg/ml]'], df_ellip_ito['Thickness [nm]'], s=200, color='w', marker='D', edgecolor=COLOURS_GREENS[800], linewidth=lw, label='on ITO, ellipsometry')

"""Axis details"""
ax.set_xlabel('TFB Concentration [mg/ml]')
ax.set_ylabel('TFB Thickness [nm]')
ax.set_ylim(bottom=-0.5, top=30.5)
ax.xaxis.set_major_locator(mpl.ticker.MultipleLocator(1))
ax.xaxis.set_minor_locator(mpl.ticker.MultipleLocator(0.5))
ax.yaxis.set_major_locator(mpl.ticker.MultipleLocator(5))
ax.yaxis.set_minor_locator(mpl.ticker.MultipleLocator(1))

"""Legend"""
leg = ax.legend(frameon = False, handletextpad=0.2, loc=[0, 0.65])


"""Graphical details"""
plt.rcParams['font.size'] = 24
plt.rc('axes', labelsize=30)    # fontsize of the x and y labels
plt.rcParams['axes.linewidth'] = lw
plt.rcParams['lines.linewidth'] = lw

ax.xaxis.set_tick_params(which='major', size=ticksize_major, width=lw)
ax.xaxis.set_tick_params(which='minor', size=ticksize_minor, width=lw)
ax.yaxis.set_tick_params(which='major', size=ticksize_major, width=lw)
ax.yaxis.set_tick_params(which='minor', size=ticksize_minor, width=lw)

"""Exporting plots"""
plot_name = 'Plot TFB Thickness'
export_name_pdf = plot_name + '.pdf'
export_name_svg = plot_name + '.svg'
export_name_png = plot_name + '.png'
#export_name_eps = plot_name + '.eps'
pdf_path = os.path.join(export_folder_path,export_name_pdf)
svg_path = os.path.join(export_folder_path,export_name_svg)
png_path = os.path.join(export_folder_path,export_name_png)
#eps_path = os.path.join(export_folder_path,export_name_eps)
if len(pdf_path) > 255:
    warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
    print(warning_string)
fig.savefig(pdf_path, bbox_inches='tight')
fig.savefig(svg_path, bbox_inches='tight')
fig.savefig(png_path, bbox_inches='tight')
#fig.savefig(eps_path, bbox_inches='tight')
plt.close(fig)

